import path from 'path'
import { fileURLToPath, URL } from 'node:url'
import { defineConfig, loadEnv } from 'vite'
import vue from '@vitejs/plugin-vue'
import vueJsx from '@vitejs/plugin-vue-jsx'
import unocss from 'unocss/vite'
import Components from 'unplugin-vue-components/vite'
import { VantResolver } from 'unplugin-vue-components/resolvers'
import { createSvgIconsPlugin } from 'vite-plugin-svg-icons'
import mockDevServerPlugin from 'vite-plugin-mock-dev-server'
import vueSetupExtend from 'vite-plugin-vue-setup-extend'
import viteCompression from 'vite-plugin-compression'
import { createHtmlPlugin } from 'vite-plugin-html'

// 当前工作目录路径
const root: string = process.cwd()

// https://vitejs.dev/config/
export default defineConfig(({ mode }) => {
  // 环境变量
  const env = loadEnv(mode, root, '')
  return {
    base: env.VITE_PUBLIC_PATH || '/',
    plugins: [
      vue(),
      vueJsx(),
      unocss(),
      mockDevServerPlugin(),
      Components({ resolvers: [VantResolver()] }), // vant 组件自动按需引入
      // svg icon
      createSvgIconsPlugin({
        iconDirs: [path.resolve(root, 'src/icons/svg')], // 指定图标文件夹
        symbolId: 'icon-[dir]-[name]' // 指定 symbolId 格式
      }),
      vueSetupExtend(), // 允许 setup 语法糖上添加组件名属性
      viteCompression(), // 生产环境 gzip 压缩资源
      createHtmlPlugin({ inject: { data: { ENABLE_ERUDA: env.VITE_ENABLE_ERUDA || 'false' } } }) // 注入模板数据
    ],
    resolve: {
      alias: {
        '@': fileURLToPath(new URL('./src', import.meta.url)),
        '~': fileURLToPath(new URL('./src/assets', import.meta.url))
      }
    },
    server: {
      host: true,
      // 仅在 proxy 中配置的代理前缀， mock-dev-server 才会拦截并 mock
      // doc: https://github.com/pengzhanbo/vite-plugin-mock-dev-server
      proxy: {
        '/api': {
          host: true,
          target: 'https://h5.hysbx.com/api',
          changeOrigin: true,
          rewrite: path => path.replace(/^\/api/, ''),
          secure: false
        }
      }
    },
    build: {
      rollupOptions: {
        output: {
          chunkFileNames: 'static/js/[name]-[hash].js',
          entryFileNames: 'static/js/[name]-[hash].js',
          assetFileNames: 'static/[ext]/[name]-[hash].[ext]'
        }
      }
    }
  }
})
